import fs from "fs/promises";
import path from "path";
import crypto from "crypto";
import { dataDir } from "./paths";

const DB_FILE = "db.json";

function nowISO() {
  return new Date().toISOString();
}

function rid(prefix) {
  return `${prefix}_${crypto.randomBytes(8).toString("hex")}`;
}

async function ensureStore() {
  const dir = dataDir();
  await fs.mkdir(dir, { recursive: true });
  const file = path.join(dir, DB_FILE);
  try {
    await fs.access(file);
  } catch {
    const initial = { events: [], guests: [], invites: [], rsvps: [] };
    await atomicWriteJson(file, initial);
  }
  return file;
}

async function atomicWriteJson(file, obj) {
  const tmp = `${file}.tmp_${crypto.randomBytes(4).toString("hex")}`;
  await fs.writeFile(tmp, JSON.stringify(obj, null, 2), "utf8");
  await fs.rename(tmp, file);
}

async function readDb() {
  const file = await ensureStore();
  const raw = await fs.readFile(file, "utf8");
  return JSON.parse(raw);
}

async function writeDb(db) {
  const file = await ensureStore();
  await atomicWriteJson(file, db);
}

export async function listEvents() {
  const db = await readDb();
  return db.events.sort((a, b) => (b.createdAt || "").localeCompare(a.createdAt || ""));
}

export async function createEvent(input) {
  const db = await readDb();
  const event = {
    id: rid("evt"),
    title: input.title || "Dasma",
    coupleA: input.coupleA,
    coupleB: input.coupleB,
    dateISO: input.dateISO,
    venueName: input.venueName,
    venueAddress: input.venueAddress,
    mapsUrl: input.mapsUrl,
    dressCode: input.dressCode || "",
    notes: input.notes || "",
    itinerary: Array.isArray(input.itinerary) ? input.itinerary : [],
    createdAt: nowISO(),
  };
  db.events.push(event);
  await writeDb(db);
  return event;
}

export async function getEvent(eventId) {
  const db = await readDb();
  return db.events.find((e) => e.id === eventId) || null;
}

export async function addGuests(eventId, guests) {
  const db = await readDb();
  const ev = db.events.find((e) => e.id === eventId);
  if (!ev) throw new Error("Event not found");

  const created = [];
  for (const g of guests) {
    const guest = {
      id: rid("gst"),
      eventId,
      name: (g.name || "").trim(),
      email: (g.email || "").trim(),
      phone: (g.phone || "").trim(),
      createdAt: nowISO(),
    };
    if (!guest.name) continue;
    db.guests.push(guest);

    const token = crypto.randomBytes(16).toString("hex");
    const invite = {
      token,
      eventId,
      guestId: guest.id,
      createdAt: nowISO(),
      sent: { email: null, sms: null, whatsapp: null },
    };
    db.invites.push(invite);

    created.push({ guest, invite });
  }

  await writeDb(db);
  return created;
}

export async function listInvites(eventId) {
  const db = await readDb();
  const guestsById = new Map(db.guests.filter(g => g.eventId === eventId).map(g => [g.id, g]));
  return db.invites
    .filter((i) => i.eventId === eventId)
    .map((i) => ({ ...i, guest: guestsById.get(i.guestId) || null }))
    .sort((a,b)=> (a.createdAt||"").localeCompare(b.createdAt||""));
}

export async function getInvite(token) {
  const db = await readDb();
  const invite = db.invites.find((i) => i.token === token);
  if (!invite) return null;
  const event = db.events.find((e) => e.id === invite.eventId) || null;
  const guest = db.guests.find((g) => g.id === invite.guestId) || null;
  const rsvp = db.rsvps.find((r) => r.inviteToken === token) || null;
  return { invite, event, guest, rsvp };
}

export async function saveRsvp(token, payload) {
  const db = await readDb();
  const invite = db.invites.find((i) => i.token === token);
  if (!invite) throw new Error("Invite not found");

  const status = payload.status; // yes | no | maybe
  const count = Number.isFinite(payload.count) ? payload.count : parseInt(payload.count || "1", 10);
  const note = (payload.note || "").toString().slice(0, 500);

  const existing = db.rsvps.find((r) => r.inviteToken === token);
  const rsvp = {
    inviteToken: token,
    status: ["yes", "no", "maybe"].includes(status) ? status : "maybe",
    count: Number.isFinite(count) && count > 0 ? Math.min(count, 10) : 1,
    note,
    updatedAt: nowISO(),
  };

  if (existing) {
    Object.assign(existing, rsvp);
  } else {
    db.rsvps.push(rsvp);
  }

  await writeDb(db);
  return rsvp;
}

export async function listRsvps(eventId) {
  const db = await readDb();
  const invites = db.invites.filter(i => i.eventId === eventId);
  const inviteByToken = new Map(invites.map(i => [i.token, i]));
  const guestsById = new Map(db.guests.filter(g => g.eventId === eventId).map(g => [g.id, g]));
  return db.rsvps
    .filter(r => inviteByToken.has(r.inviteToken))
    .map(r => {
      const inv = inviteByToken.get(r.inviteToken);
      const guest = guestsById.get(inv.guestId) || null;
      return { ...r, token: r.inviteToken, guest };
    })
    .sort((a,b)=> (b.updatedAt||"").localeCompare(a.updatedAt||""));
}
