"use client";

import { useMemo, useState } from "react";
import Link from "next/link";

function getKey() {
  try { return localStorage.getItem("ADMIN_KEY") || ""; } catch { return ""; }
}

export default function NewEvent() {
  const [saving, setSaving] = useState(false);
  const [err, setErr] = useState("");
  const key = useMemo(() => getKey(), []);

  async function submit(e) {
    e.preventDefault();
    setSaving(true);
    setErr("");
    try {
      const fd = new FormData(e.currentTarget);
      const itinerary = [
        { label: "Mbërritja", time: fd.get("t1") || "19:00" },
        { label: "Ceremonia", time: fd.get("t2") || "19:30" },
        { label: "Darka", time: fd.get("t3") || "20:30" },
      ];

      const payload = {
        coupleA: fd.get("coupleA"),
        coupleB: fd.get("coupleB"),
        dateISO: new Date(`${fd.get("date")}T${fd.get("time")}:00`).toISOString(),
        venueName: fd.get("venueName"),
        venueAddress: fd.get("venueAddress"),
        mapsUrl: fd.get("mapsUrl"),
        dressCode: fd.get("dressCode"),
        notes: fd.get("notes"),
        itinerary,
      };

      const res = await fetch("/api/events", {
        method: "POST",
        headers: { "content-type": "application/json", "x-admin-key": key },
        body: JSON.stringify(payload),
      });

      if (!res.ok) {
        setErr("S'u krijua event. Kontrollo ADMIN_KEY.");
        return;
      }

      const json = await res.json();
      window.location.href = `/admin/events/${json.event.id}`;
    } finally {
      setSaving(false);
    }
  }

  return (
    <main className="min-h-screen bg-neutral-950 text-white">
      <div className="mx-auto max-w-3xl px-6 py-10">
        <div className="flex items-center justify-between gap-4">
          <div>
            <div className="text-xs tracking-widest text-neutral-400">EVENT I RI</div>
            <h1 className="mt-2 text-2xl font-semibold">Krijo dasmen</h1>
            {err ? <div className="mt-2 text-sm text-red-300">{err}</div> : null}
          </div>
          <Link href="/admin" className="rounded-xl border border-white/10 px-4 py-2 text-sm text-neutral-200 hover:bg-white/5">
            Mbrapa
          </Link>
        </div>

        <form onSubmit={submit} className="mt-8 space-y-6">
          <Section title="Cifti">
            <div className="grid gap-4 md:grid-cols-2">
              <Field label="Emri 1" name="coupleA" placeholder="Arlind" required />
              <Field label="Emri 2" name="coupleB" placeholder="Iris" required />
            </div>
          </Section>

          <Section title="Data & Ora">
            <div className="grid gap-4 md:grid-cols-2">
              <Field label="Data" name="date" type="date" required />
              <Field label="Ora" name="time" type="time" required />
            </div>
          </Section>

          <Section title="Lokacioni">
            <Field label="Emri i vendit" name="venueName" placeholder="Restaurant ..." required />
            <Field label="Adresa" name="venueAddress" placeholder="Rr. ..." required />
            <Field label="Link i hartes (Google Maps)" name="mapsUrl" placeholder="https://maps.google.com/?q=..." />
          </Section>

          <Section title="Detaje">
            <Field label="Dress code" name="dressCode" placeholder="Elegant / Formal / ..." />
            <Field label="Shenime" name="notes" placeholder="P.sh. parkim, hyrje, etj." />
          </Section>

          <Section title="Itinerari (opsional)">
            <div className="grid gap-4 md:grid-cols-3">
              <Field label="Koha 1" name="t1" placeholder="19:00" />
              <Field label="Koha 2" name="t2" placeholder="19:30" />
              <Field label="Koha 3" name="t3" placeholder="20:30" />
            </div>
          </Section>

          <button
            disabled={saving}
            className="w-full rounded-xl bg-white px-4 py-3 text-sm font-semibold text-black hover:bg-neutral-200 disabled:opacity-60"
          >
            {saving ? "Duke ruajtur…" : "Krijo eventin"}
          </button>
        </form>
      </div>
    </main>
  );
}

function Section({ title, children }) {
  return (
    <div className="rounded-3xl border border-white/10 bg-white/5 p-6">
      <div className="text-base font-semibold">{title}</div>
      <div className="mt-4">{children}</div>
    </div>
  );
}

function Field({ label, name, placeholder, type = "text", required }) {
  return (
    <label className="block">
      <div className="text-xs text-neutral-400">{label}</div>
      <input
        name={name}
        type={type}
        placeholder={placeholder}
        required={required}
        className="mt-2 w-full rounded-xl border border-white/10 bg-black/40 px-3 py-2 text-sm text-white outline-none focus:border-white/25"
      />
    </label>
  );
}
